/*
 * bdclLayerManager.cpp
 *
 *  Created on: Jan 2, 2017
 *      Author: pgovindaraju
 */

#include "bdclLayerManager.h"

LOG_IMPORT_CONTEXT(tbdcl)

namespace adit { namespace bdcl
{
using namespace std;

bdclLayerManager::bdclLayerManager(void){
    LOG_INFO((tbdcl, "inside the LayerManager constructor"));
    mLayerId = LAYER_ID;
    mDisplayId = DISPLAY_ID;
    mSurfaceId = SURFACE_ID;
    mMaxDevices = DEVICE_MAX;
    mIlmInit = false;
}

bdclLayerManager::~bdclLayerManager()
{
}

bool bdclLayerManager::init()
{
    if (!mIlmInit)
    {
        if (ILM_SUCCESS != ilm_init())
        {
            LOG_ERROR((tbdcl, "ilm_init failed"));
            return false;
        }
    }

    mIlmInit = true;

    t_ilm_display* display_ids = NULL;
    t_ilm_uint display_count;
    ilm_getScreenIDs(&display_count, &display_ids);
    for(unsigned int i = 0; i < display_count; i++)
    {
        // 0 = HDMI
        if(display_ids[i] == 0){
            mDisplayId = 0;
        }
    }

    return createLayer(RES_WIDTH, RES_HEIGHT);
}

bool bdclLayerManager::createLayer(const t_ilm_uint inSrcWidth, const t_ilm_uint inSrcHeight)
{
    if (ILM_SUCCESS != ilm_registerNotification(ilmNotifyCB, this))
    {
        LOG_ERROR((tbdcl, "%s()  ilm_registerNotification failed", __func__));
        return false;
    }

    if (ILM_SUCCESS != ilm_layerCreateWithDimension(&mLayerId, inSrcWidth, inSrcHeight))
    {
        LOG_ERROR((tbdcl, "ilm_layerCreateWithDimension for layer %d", mLayerId));
        return false;
    }

    t_ilm_uint screen_width, screen_height;
    if (ILM_SUCCESS != ilm_getScreenResolution(mDisplayId, &screen_width, &screen_height))
    {
        LOG_ERROR((tbdcl, "ilm_getScreenResolution for screen %d", mDisplayId));
        return false;
    }

    t_ilm_uint origin_x = 0, origin_y = 0;
    if(screen_width > inSrcWidth)
    {
        origin_x = (screen_width - inSrcWidth) / 2;
    }
    if(screen_height > inSrcHeight)
    {
        origin_y = (screen_height - inSrcHeight) / 2;
    }

    if (ILM_SUCCESS != ilm_layerSetDestinationRectangle(mLayerId, origin_x, origin_y,
                                                        inSrcWidth, inSrcHeight))
    {
        LOG_ERROR((tbdcl, "layerSetDestinationRectangle for layer %d", mLayerId));
        return false;
    }

    if (ILM_SUCCESS != ilm_layerSetVisibility(mLayerId, ILM_TRUE))
    {
        LOG_ERROR((tbdcl, "ilm_layerSetVisibility failed for layer %d", mLayerId));
        return false;
    }

    if (ILM_SUCCESS != ilm_displaySetRenderOrder(mDisplayId, &mLayerId, 1))
    {
        LOG_ERROR((tbdcl, "ilm_displaySetRenderOrder failed for display %d, layer %d", mDisplayId,
                mLayerId));
        return false;
    }

    if (ILM_SUCCESS != ilm_commitChanges())
    {
        LOG_ERROR((tbdcl, "ilm_commitChanges failed"));
        return false;
    }

    return true;
}

void bdclLayerManager::finalize()
{
    if (mIlmInit)
    {
        LOG_INFO((tbdcl, "finalize layer setting"));

        removeSurface(mSurfaceId);

        removeLayer();
        ilm_unregisterNotification();
        ilm_destroy();

        mIlmInit = false;
    }
}
void bdclLayerManager::removeSurface(const t_ilm_surface inSurfaceId)
{
    LOGD_DEBUG((tbdcl, "remove surface %d", inSurfaceId));
    ilm_surfaceRemove(inSurfaceId);

    if (ILM_SUCCESS != ilm_commitChanges())
    {
        LOG_ERROR((tbdcl, "ilm_commitChanges failed"));
    }
}

void bdclLayerManager::removeLayer()
{
    if (ILM_SUCCESS != ilm_layerSetVisibility(mLayerId, ILM_FALSE))
    {
        LOG_ERROR((tbdcl, "ilm_layerSetVisibility failed for layer %d", mLayerId));
    }
    if (ILM_SUCCESS != ilm_commitChanges())
    {
        LOG_ERROR((tbdcl, "ilm_commitChanges failed for ilm_layerSetVisibility"));
    }

    LOGD_DEBUG((tbdcl, "remove layer %d", mLayerId));
    ilm_layerRemove(mLayerId);

    if (ILM_SUCCESS != ilm_commitChanges())
    {
        LOG_ERROR((tbdcl, "ilm_commitChanges failed"));
    }
}

void bdclLayerManager::ilmNotifyCB(ilmObjectType object, t_ilm_uint id, t_ilm_bool created, void *user_data)
{
    LOG_INFO((tbdcl, "bdclLayerManager::%s() called ", __FUNCTION__));

    bdclLayerManager *bdcllm = static_cast<bdclLayerManager*> (user_data);
    struct ilmSurfaceProperties sp;

    if (object == ILM_SURFACE)
    {
        if (created)
        {
            LOG_INFO((tbdcl,"bdclLayerManager::%s() surface (id=%d) created", __FUNCTION__, id));
            ilm_getPropertiesOfSurface(id, &sp);

            LOG_INFO((tbdcl,"bdclLayerManager::%s() origSourceWidth: %d  origSourceHeight: %d",
                    __FUNCTION__, sp.origSourceWidth, sp.origSourceHeight));
            if ((sp.origSourceWidth != 0) && (sp.origSourceHeight != 0))
            {
                    // surface is already configured
                    bdcllm->configure_ilm_surface(id, sp.origSourceWidth, sp.origSourceHeight);
            }
            else
            {
                LOG_WARN((tbdcl,"bdclLayerManager::%s() surface (id=%d) not configured. Configure notification not supported on IMX6",
                        __FUNCTION__, id));
            }
        }
        else if(!created)
        {
            LOG_INFO((tbdcl,"bdclLayerManager::%s() surface (id=%d) destroyed", __FUNCTION__, id));
        }
    }
    else if (object == ILM_LAYER)
    {
        if (created)
            LOG_INFO((tbdcl,"bdclLayerManager::%s() layer (id=%d) created", __FUNCTION__, id));
        else if(!created)
            LOG_INFO((tbdcl,"bdclLayerManager::%s() layer (id=%d) destroyed", __FUNCTION__, id));
    }
}

void bdclLayerManager::configure_ilm_surface(t_ilm_uint id, t_ilm_uint width, t_ilm_uint height)
{
    LOG_INFO((tbdcl, "bdclLayerManager::%s() called", __FUNCTION__));

    if (ILM_SUCCESS != ilm_surfaceSetDestinationRectangle(id, 0, 0, width, height))
        LOG_ERROR((tbdcl, "bdclLayerManager::%s() ilm_surfaceSetDestinationRectangle failed for surface %d ", __FUNCTION__, id));
    if (ILM_SUCCESS != ilm_surfaceSetSourceRectangle(id, 0, 0, width, height))
        LOG_ERROR((tbdcl, "bdclLayerManager::%s() ilm_surfaceSetSourceRectangle failed for surface %d ", __FUNCTION__, id));
    if (ILM_SUCCESS != ilm_surfaceSetVisibility(id, ILM_TRUE))
        LOG_ERROR((tbdcl, "bdclLayerManager::%s() ilm_surfaceSetVisibility failed for surface %d ", __FUNCTION__, id));
    if (ILM_SUCCESS != ilm_layerAddSurface(mLayerId,id))
        LOG_ERROR((tbdcl, "bdclLayerManager::%s() ilm_layerAddSurface failed for surface %d ", __FUNCTION__, id));
    else
        LOG_INFO((tbdcl,"bdclLayerManager::%s() surface ID (%d) is added to layer ID (%d)", __FUNCTION__, id, mLayerId));

    ilm_commitChanges();
}

} }
